#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Basic but functional MP3/WAV player with GUI.
Features:
- Drag-and-drop to add songs (via tkinterdnd2 if installed; falls back to File > Open Files)
- Shuffle toggle
- Volume control
- Double-click a song to play
- Next / Previous / Play-Pause / Stop
- Save / Open playlist in a custom .ahpl (JSON) format
- Auto-saves last session and restores on launch

Dependencies:
    pip install pygame
    # Optional for OS drag-and-drop:
    pip install tkinterdnd2

Run:
    python anthro_simple_player.py
"""

import os
import sys
import json
import random
import threading
import time
import tkinter as tk
from tkinter import ttk, filedialog, messagebox

# Try to import tkinterdnd2 for drag-and-drop
try:
    from tkinterdnd2 import DND_FILES, TkinterDnD
    DND_AVAILABLE = True
except Exception:
    DND_AVAILABLE = False
    DND_FILES = None
    TkinterDnD = None

# Try to import pygame for audio playback
try:
    import pygame
    PYGAME_AVAILABLE = True
except Exception:
    PYGAME_AVAILABLE = False

SUPPORTED_EXTS = {".mp3", ".wav"}

APP_NAME = "Anthro Simple Player"
APP_ID = "anthro_simple_player"
PLAYLIST_EXT = ".ahpl"  # Anthro playlist
CONFIG_DIR = os.path.join(os.path.expanduser("~"), f".{APP_ID}")
AUTO_PATH = os.path.join(CONFIG_DIR, f"last_playlist{PLAYLIST_EXT}")


def ensure_config_dir():
    try:
        os.makedirs(CONFIG_DIR, exist_ok=True)
    except Exception:
        pass


def is_audio_file(path: str) -> bool:
    return os.path.isfile(path) and os.path.splitext(path)[1].lower() in SUPPORTED_EXTS


def find_audio_files_in(path: str):
    """Yield supported audio files from file or directory (recursive)."""
    path = os.path.normpath(path)
    if os.path.isfile(path) and is_audio_file(path):
        yield path
    elif os.path.isdir(path):
        for root, _dirs, files in os.walk(path):
            for f in files:
                p = os.path.join(root, f)
                if is_audio_file(p):
                    yield p


def parse_dnd_event_data(data: str):
    """
    Tk DND gives a string like:
      '{C:/Users/me/Music/track 1.mp3} {/Users/me/file with spaces.wav} /path/no_spaces.mp3'
    Return a clean list of file paths.
    """
    if not data:
        return []
    # Split on spaces, but respect braces
    out = []
    buf = ""
    in_brace = False
    for ch in data:
        if ch == "{":
            in_brace = True
            if buf.strip():
                out.append(buf.strip())
                buf = ""
            continue
        if ch == "}":
            in_brace = False
            if buf:
                out.append(buf)
                buf = ""
            continue
        if ch == " " and not in_brace:
            if buf.strip():
                out.append(buf.strip())
                buf = ""
            continue
        buf += ch
    if buf.strip():
        out.append(buf.strip())
    return out


class Playlist:
    def __init__(self):
        self.tracks = []  # list of absolute file paths
        self.current_index = -1
        self.shuffle = False
        self.volume = 0.8
        self._shuffle_order = []
        self._shuffle_pos = -1

    # --- Core ops ---
    def add_paths(self, paths):
        before = set(self.tracks)
        added = 0
        for p in paths:
            ap = os.path.abspath(p)
            if ap not in before and is_audio_file(ap):
                self.tracks.append(ap)
                added += 1
        return added

    def clear(self):
        self.tracks.clear()
        self.current_index = -1
        self._shuffle_order.clear()
        self._shuffle_pos = -1

    def ensure_valid_index(self):
        if not self.tracks:
            self.current_index = -1
        else:
            if self.current_index < 0 or self.current_index >= len(self.tracks):
                self.current_index = 0

    def set_shuffle(self, enabled: bool):
        if self.shuffle == enabled:
            return
        self.shuffle = enabled
        if enabled:
            self._build_shuffle_order()
        else:
            self._shuffle_order.clear()
            self._shuffle_pos = -1

    def _build_shuffle_order(self):
        n = len(self.tracks)
        self._shuffle_order = list(range(n))
        random.shuffle(self._shuffle_order)
        # Try to start from current track if set
        if 0 <= self.current_index < n:
            try:
                pos = self._shuffle_order.index(self.current_index)
                self._shuffle_order[0], self._shuffle_order[pos] = self._shuffle_order[pos], self._shuffle_order[0]
            except ValueError:
                pass
        self._shuffle_pos = 0

    def next_index(self):
        if not self.tracks:
            return -1
        if self.shuffle:
            if not self._shuffle_order:
                self._build_shuffle_order()
            self._shuffle_pos += 1
            if self._shuffle_pos >= len(self._shuffle_order):
                # Reshuffle for a fresh cycle
                self._build_shuffle_order()
                self._shuffle_pos = 0
            return self._shuffle_order[self._shuffle_pos]
        # linear
        return (self.current_index + 1) % len(self.tracks)

    def prev_index(self):
        if not self.tracks:
            return -1
        if self.shuffle:
            if not self._shuffle_order:
                self._build_shuffle_order()
            self._shuffle_pos -= 1
            if self._shuffle_pos < 0:
                self._shuffle_pos = len(self._shuffle_order) - 1
            return self._shuffle_order[self._shuffle_pos]
        # linear
        return (self.current_index - 1) % len(self.tracks)

    # --- Persistence ---
    def to_dict(self):
        return {
            "version": 1,
            "tracks": self.tracks,
            "current_index": self.current_index,
            "shuffle": self.shuffle,
            "volume": self.volume,
        }

    def from_dict(self, data):
        tracks = [p for p in data.get("tracks", []) if os.path.exists(p) and is_audio_file(p)]
        self.tracks = tracks
        self.current_index = int(data.get("current_index", -1))
        self.shuffle = bool(data.get("shuffle", False))
        self.volume = float(data.get("volume", 0.8))
        if self.shuffle:
            self._build_shuffle_order()
        else:
            self._shuffle_order.clear()
            self._shuffle_pos = -1


class PlayerApp:
    def __init__(self):
        if DND_AVAILABLE:
            self.root = TkinterDnD.Tk()
        else:
            self.root = tk.Tk()
        self.root.title(APP_NAME)
        self.root.geometry("720x480")

        if not PYGAME_AVAILABLE:
            messagebox.showwarning(
                "Missing dependency",
                "The 'pygame' package is required for audio playback.\n\nInstall it with:\n    pip install pygame"
            )

        self.playlist = Playlist()
        self.playing = False
        self.paused = False
        self.playlist_path = None  # last saved/loaded .ahpl path

        # Initialize pygame mixer
        if PYGAME_AVAILABLE:
            try:
                pygame.mixer.init(frequency=44100, size=-16, channels=2, buffer=1024)
                pygame.mixer.music.set_volume(self.playlist.volume)
            except Exception as e:
                messagebox.showerror("Audio init failed", f"pygame.mixer init failed:\n{e}")

        self._build_ui()
        ensure_config_dir()
        self._load_autosave()

        self.root.protocol("WM_DELETE_WINDOW", self.on_close)
        # Poll for track end
        self._poll_playback_end()

    # --- UI ---
    def _build_ui(self):
        self._build_menu()

        outer = ttk.Frame(self.root, padding=8)
        outer.pack(fill="both", expand=True)

        # Controls
        controls = ttk.Frame(outer)
        controls.pack(fill="x")

        self.btn_prev = ttk.Button(controls, text="⏮ Prev", width=8, command=self.on_prev)
        self.btn_play = ttk.Button(controls, text="▶ Play", width=8, command=self.on_play_pause)
        self.btn_stop = ttk.Button(controls, text="⏹ Stop", width=8, command=self.on_stop)
        self.btn_next = ttk.Button(controls, text="⏭ Next", width=8, command=self.on_next)

        self.btn_prev.pack(side="left", padx=(0, 6))
        self.btn_play.pack(side="left", padx=6)
        self.btn_stop.pack(side="left", padx=6)
        self.btn_next.pack(side="left", padx=6)

        self.shuffle_var = tk.BooleanVar(value=False)
        self.chk_shuffle = ttk.Checkbutton(controls, text="Shuffle", variable=self.shuffle_var, command=self.toggle_shuffle)
        self.chk_shuffle.pack(side="left", padx=12)

        ttk.Label(controls, text="Volume").pack(side="left", padx=(12, 4))
        self.vol_var = tk.DoubleVar(value=80.0)
        self.scale_vol = ttk.Scale(controls, from_=0, to=100, variable=self.vol_var, command=self.on_volume_change, length=160)
        self.scale_vol.pack(side="left", padx=(0, 6))

        # Drop hint
        hint_frame = ttk.Frame(outer)
        hint_frame.pack(fill="x", pady=(8, 4))
        drop_hint = "Drag & drop files/folders here" if DND_AVAILABLE else "Drag & drop requires 'tkinterdnd2'. Use File → Open Files…"
        self.lbl_hint = ttk.Label(hint_frame, text=drop_hint, foreground="#666")
        self.lbl_hint.pack(side="left")

        # Playlist listbox with scrollbar
        list_frame = ttk.Frame(outer)
        list_frame.pack(fill="both", expand=True, pady=(4, 4))

        self.listbox = tk.Listbox(list_frame, selectmode="browse", activestyle="dotbox")
        self.listbox.pack(side="left", fill="both", expand=True)
        sb = ttk.Scrollbar(list_frame, orient="vertical", command=self.listbox.yview)
        sb.pack(side="right", fill="y")
        self.listbox.config(yscrollcommand=sb.set)

        self.listbox.bind("<Double-Button-1>", self.on_double_click)
        self.listbox.bind("<Return>", self.on_enter_key)

        if DND_AVAILABLE:
            self.listbox.drop_target_register(DND_FILES)
            self.listbox.dnd_bind("<<Drop>>", self.on_drop)

        # Status bar
        self.status_var = tk.StringVar(value="Ready")
        self.lbl_status = ttk.Label(outer, textvariable=self.status_var, anchor="w")
        self.lbl_status.pack(fill="x", pady=(6, 0))

    def _build_menu(self):
        menubar = tk.Menu(self.root)
        self.root.config(menu=menubar)

        m_file = tk.Menu(menubar, tearoff=False)
        menubar.add_cascade(label="File", menu=m_file)
        m_file.add_command(label="New Playlist", command=self.new_playlist, accelerator="Ctrl+N")
        m_file.add_separator()
        m_file.add_command(label="Open Files…", command=self.open_files, accelerator="Ctrl+O")
        m_file.add_command(label="Open Playlist…", command=self.open_playlist, accelerator="Ctrl+Shift+O")
        m_file.add_separator()
        m_file.add_command(label="Save Playlist", command=self.save_playlist, accelerator="Ctrl+S")
        m_file.add_command(label="Save Playlist As…", command=self.save_playlist_as, accelerator="Ctrl+Shift+S")
        m_file.add_separator()
        m_file.add_command(label="Exit", command=self.on_close, accelerator="Ctrl+Q")

        # Shortcuts
        self.root.bind_all("<Control-n>", lambda e: self.new_playlist())
        self.root.bind_all("<Control-o>", lambda e: self.open_files())
        self.root.bind_all("<Control-Shift-O>", lambda e: self.open_playlist())
        self.root.bind_all("<Control-s>", lambda e: self.save_playlist())
        self.root.bind_all("<Control-Shift-S>", lambda e: self.save_playlist_as())
        self.root.bind_all("<Control-q>", lambda e: self.on_close())

    # --- Playlist ops ---
    def refresh_listbox(self):
        self.listbox.delete(0, "end")
        for i, p in enumerate(self.playlist.tracks):
            name = os.path.basename(p)
            self.listbox.insert("end", f"{i+1:02d}. {name}")
        if 0 <= self.playlist.current_index < len(self.playlist.tracks):
            self.listbox.selection_clear(0, "end")
            self.listbox.selection_set(self.playlist.current_index)
            self.listbox.see(self.playlist.current_index)

    def add_files(self, paths):
        # Expand folders into audio files
        all_files = []
        for p in paths:
            all_files.extend(find_audio_files_in(p))
        added = self.playlist.add_paths(all_files)
        if added:
            self.status(f"Added {added} track(s).")
        else:
            self.status("No new audio files found.")
        self.refresh_listbox()
        self._autosave_silent()

    def on_drop(self, event):
        paths = parse_dnd_event_data(event.data)
        if not paths:
            return
        self.add_files(paths)

    def on_double_click(self, _event=None):
        idx = self.listbox.curselection()
        if not idx:
            return
        self.play_index(idx[0])

    def on_enter_key(self, _event=None):
        self.on_double_click()

    def new_playlist(self):
        if self.playlist.tracks and not messagebox.askyesno("New Playlist", "Clear current playlist?"):
            return
        self.stop_audio()
        self.playlist.clear()
        self.playlist_path = None
        self.refresh_listbox()
        self.status("New playlist.")
        self._autosave_silent()

    def open_files(self):
        paths = filedialog.askopenfilenames(
            title="Select audio files",
            filetypes=[("Audio", "*.mp3 *.wav"), ("All files", "*.*")],
        )
        if paths:
            self.add_files(paths)

    def open_playlist(self):
        path = filedialog.askopenfilename(
            title="Open playlist",
            filetypes=[("Anthro Playlist", f"*{PLAYLIST_EXT}"), ("All files", "*.*")],
        )
        if not path:
            return
        try:
            with open(path, "r", encoding="utf-8") as f:
                data = json.load(f)
            self.stop_audio()
            self.playlist.from_dict(data)
            self.playlist_path = path
            self.refresh_listbox()
            self.shuffle_var.set(self.playlist.shuffle)
            self.scale_vol.set(self.playlist.volume * 100.0)
            if self.playlist.current_index >= 0 and self.playlist.current_index < len(self.playlist.tracks):
                self.status(f"Loaded playlist: {os.path.basename(path)} ({len(self.playlist.tracks)} tracks)")
            else:
                self.status(f"Loaded playlist: {os.path.basename(path)} (empty)")
        except Exception as e:
            messagebox.showerror("Open failed", f"Could not open playlist:\n{e}")

    def save_playlist(self):
        if not self.playlist.tracks:
            messagebox.showinfo("Nothing to save", "Playlist is empty.")
            return
        if self.playlist_path is None:
            return self.save_playlist_as()
        try:
            data = self.playlist.to_dict()
            with open(self.playlist_path, "w", encoding="utf-8") as f:
                json.dump(data, f, indent=2)
            self.status(f"Saved playlist: {os.path.basename(self.playlist_path)}")
        except Exception as e:
            messagebox.showerror("Save failed", f"Could not save playlist:\n{e}")

    def save_playlist_as(self):
        if not self.playlist.tracks:
            messagebox.showinfo("Nothing to save", "Playlist is empty.")
            return
        path = filedialog.asksaveasfilename(
            title="Save playlist as",
            defaultextension=PLAYLIST_EXT,
            filetypes=[("Anthro Playlist", f"*{PLAYLIST_EXT}"), ("All files", "*.*")],
            initialfile="playlist" + PLAYLIST_EXT,
        )
        if not path:
            return
        self.playlist_path = path
        self.save_playlist()

    # --- Playback ---
    def play_index(self, index: int):
        if not PYGAME_AVAILABLE:
            messagebox.showwarning("pygame required", "Install pygame to play audio:\n\npip install pygame")
            return
        if not (0 <= index < len(self.playlist.tracks)):
            return
        path = self.playlist.tracks[index]
        try:
            pygame.mixer.music.load(path)
            pygame.mixer.music.play()
            pygame.mixer.music.set_volume(self.playlist.volume)
            self.playlist.current_index = index
            self.playing = True
            self.paused = False
            self.btn_play.config(text="⏸ Pause")
            self._highlight_current()
            self.status_now_playing()
            self._autosave_silent()
        except Exception as e:
            messagebox.showerror("Play failed", f"Could not play file:\n{path}\n\n{e}")

    def on_play_pause(self):
        if not PYGAME_AVAILABLE:
            messagebox.showwarning("pygame required", "Install pygame to play audio:\n\npip install pygame")
            return
        if not self.playing:
            # Start with selected or first
            if self.playlist.current_index == -1 and self.playlist.tracks:
                self.play_index(0)
            elif self.playlist.current_index >= 0:
                # Resume current track
                self.play_index(self.playlist.current_index)
        else:
            if not self.paused:
                pygame.mixer.music.pause()
                self.paused = True
                self.btn_play.config(text="▶ Resume")
                self.status("Paused")
            else:
                pygame.mixer.music.unpause()
                self.paused = False
                self.btn_play.config(text="⏸ Pause")
                self.status_now_playing()

    def on_stop(self):
        self.stop_audio()
        self.status("Stopped")

    def on_next(self):
        if not self.playlist.tracks:
            return
        idx = self.playlist.next_index()
        if idx != -1:
            self.play_index(idx)

    def on_prev(self):
        if not self.playlist.tracks:
            return
        idx = self.playlist.prev_index()
        if idx != -1:
            self.play_index(idx)

    def stop_audio(self):
        if PYGAME_AVAILABLE:
            try:
                pygame.mixer.music.stop()
            except Exception:
                pass
        self.playing = False
        self.paused = False
        self.btn_play.config(text="▶ Play")
        self._highlight_current(clear_only=True)

    def _poll_playback_end(self):
        # Called periodically to detect when a track finishes
        if PYGAME_AVAILABLE and self.playing and not self.paused:
            try:
                busy = pygame.mixer.music.get_busy()
            except Exception:
                busy = False
            if not busy:
                # Track ended, advance
                if self.playlist.tracks:
                    idx = self.playlist.next_index()
                    if idx != -1:
                        self.play_index(idx)
        self.root.after(500, self._poll_playback_end)

    def _highlight_current(self, clear_only=False):
        self.listbox.selection_clear(0, "end")
        if not clear_only and 0 <= self.playlist.current_index < len(self.playlist.tracks):
            self.listbox.selection_set(self.playlist.current_index)
            self.listbox.see(self.playlist.current_index)

    def status(self, text: str):
        self.status_var.set(text)

    def status_now_playing(self):
        idx = self.playlist.current_index
        if 0 <= idx < len(self.playlist.tracks):
            name = os.path.basename(self.playlist.tracks[idx])
            self.status_var.set(f"Now playing [{idx+1}/{len(self.playlist.tracks)}]: {name}")

    def on_volume_change(self, _val=None):
        vol = max(0.0, min(1.0, float(self.vol_var.get()) / 100.0))
        self.playlist.volume = vol
        if PYGAME_AVAILABLE:
            try:
                pygame.mixer.music.set_volume(vol)
            except Exception:
                pass
        # no spammy status updates

    def toggle_shuffle(self):
        enabled = bool(self.shuffle_var.get())
        self.playlist.set_shuffle(enabled)
        self.status("Shuffle: ON" if enabled else "Shuffle: OFF")
        self._autosave_silent()

    # --- Autosave ---
    def _autosave_silent(self):
        ensure_config_dir()
        try:
            with open(AUTO_PATH, "w", encoding="utf-8") as f:
                json.dump(self.playlist.to_dict(), f, indent=2)
        except Exception:
            pass

    def _load_autosave(self):
        if not os.path.exists(AUTO_PATH):
            return
        try:
            with open(AUTO_PATH, "r", encoding="utf-8") as f:
                data = json.load(f)
            self.playlist.from_dict(data)
            self.refresh_listbox()
            self.shuffle_var.set(self.playlist.shuffle)
            self.scale_vol.set(self.playlist.volume * 100.0)
            if self.playlist.tracks:
                self.status(f"Restored last session ({len(self.playlist.tracks)} tracks).")
            else:
                self.status("Restored last session (empty).")
        except Exception:
            pass

    def on_close(self):
        self._autosave_silent()
        self.root.destroy()


def main():
    app = PlayerApp()
    app.root.mainloop()


if __name__ == "__main__":
    main()
